<?php
// Include database configuration
require_once '../config/db_config.php';
require_once '../models/Product.php';

// Check if user is logged in and is admin
session_start();
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: login.php');
    exit;
}

$db = getDbConnection();

// Check if database connection is available
if ($db === null) {
    die("Koneksi database gagal. Silakan coba lagi nanti atau hubungi administrator.");
}

// Get statistics
try {
    // Total orders
    $stmt = $db->query("SELECT COUNT(*) as total FROM orders");
    $total_orders = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Total revenue
    $stmt = $db->query("SELECT SUM(total_amount) as total FROM orders WHERE status = 'paid'");
    $total_revenue = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
    
    // Pending payment proofs
    $stmt = $db->query("SELECT COUNT(*) as total FROM payment_proofs WHERE status = 'pending'");
    $pending_proofs = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Recent orders
    $stmt = $db->query("
        SELECT o.*, COUNT(oi.item_id) as total_items 
        FROM orders o 
        LEFT JOIN order_items oi ON o.order_id = oi.order_id 
        GROUP BY o.order_id 
        ORDER BY o.created_at DESC 
        LIMIT 5
    ");
    $recent_orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Admin dashboard database error: " . $e->getMessage());
    $error_message = "Terjadi kesalahan sistem. Silakan coba lagi nanti.";
    $total_orders = 0;
    $total_revenue = 0;
    $pending_proofs = 0;
    $recent_orders = [];
}

// Get products from database
try {
    $product = new Product();
    $products = $product->getAllProducts();
} catch (Exception $e) {
    error_log("Admin dashboard product error: " . $e->getMessage());
    $products = [];
    $error_message = "Error loading products: " . $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Baso Alam</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .sidebar {
            min-height: 100vh;
            background-color: #343a40;
            padding-top: 20px;
        }
        .sidebar a {
            color: #fff;
            text-decoration: none;
            padding: 10px 20px;
            display: block;
        }
        .sidebar a:hover {
            background-color: #495057;
        }
        .sidebar a.active {
            background-color: #0d6efd;
        }
        .stat-card {
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .stat-card i {
            font-size: 2rem;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <div class="col-lg-2 p-0">
                <?php include 'admin_sidebar.php'; ?>
            </div>
            <div class="col-lg-10 p-4">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>Dashboard</h2>
                    <div>
                        <span class="me-3">Welcome<?php echo isset($_SESSION['name']) && $_SESSION['name'] ? ', ' . htmlspecialchars($_SESSION['name']) : ''; ?></span>
                    </div>
                </div>

                <!-- Statistics Cards -->
                <div class="row">
                    <div class="col-md-4">
                        <div class="stat-card bg-primary text-white">
                            <i class="fas fa-shopping-cart"></i>
                            <h3><?php echo number_format($total_orders); ?></h3>
                            <p class="mb-0">Total Pesanan</p>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="stat-card bg-success text-white">
                            <i class="fas fa-money-bill-wave"></i>
                            <h3>Rp <?php echo number_format($total_revenue, 0, ',', '.'); ?></h3>
                            <p class="mb-0">Total Pendapatan</p>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="stat-card bg-warning text-dark">
                            <i class="fas fa-clock"></i>
                            <h3><?php echo number_format($pending_proofs); ?></h3>
                            <p class="mb-0">Bukti Pembayaran Pending</p>
                        </div>
                    </div>
                </div>

             
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html> 